/******************************************************************************
File:       eloApi.h
Purpose:    Windows CE. Declares classes for applications to communicate with 
Elo touchscreen drivers. These are wrappers for classes declared in eloIoctl, 
through which applications and drivers exchange information. Drivers do not 
include eloApi.h.

Copyright 2008 Elo TouchSystems
All rights reserved

Updated:    2008-09-04
******************************************************************************/
#ifndef ELOAPI_H
#define ELOAPI_H

#include <eloIoctl.h>

#ifdef EloApi_Static
	#define T_EloApi
#else
	#ifdef EloApi_Export
		#define T_EloApi __declspec(dllexport)
	#else 
		#define T_EloApi __declspec(dllimport)
	#endif
#endif

#define MAX_ELO_DEVS 4
#define MAX_ELO_NAME_LEN 11

class T_EloApi EloId
{
public:
    TCHAR   mDevName[ 12 ];
    UCHAR   mDevNum;
    EloId( void ) { mDevNum = 0; memset( mDevName, 0, sizeof( mDevName )); }
};

class T_EloApi EloDevApiHdr : public EloId, public IocHdr 
{
public:
    EloDevApiHdr( DWORD ioctl = 0 ) : IocHdr() {}
};

enum { DEVSTAT_NEXIST = 0, DEVSTAT_OK, DEVSTAT_FAIL }; // EloDevApi.mStat values

class T_EloApi EloDevApi : public EloId
{
public:
    USHORT  mLink; // DevLink = LINK_UNKNOWN, LINK_SERIAL, or LINK_USB
    int     mStat; // DEVSTAT_OK, DEVSTAT_NEXIST
	UCHAR	mClass; // e.g. 32 = 2701. See smartSetCmd.h. See SmartSet mode command.
    bool    isActive( void ) { return mStat == DEVSTAT_OK; }
};

class T_EloApi EloDevId
{
public:
    TCHAR   mDevName[ MAX_ELO_NAME_LEN + 1 ];
    UCHAR   mDevNum;
    DevLink mLink;

    EloDevId( UCHAR devNum = 0, TCHAR *name = 0 );
    void	makeNameFromNum( void );
    void	setDevice( UCHAR devNum, TCHAR *name );
    ELO_ERR openDevice( void );
};
// EloDevId::EloDevId and EloDevId::setDevice assign devNum to mDevNum 
// regardless of the value. If name is NULL then, if devNum < 1, mDevName[0] is 
// assigned 0 but, if devNum >= 1, mDevName is assigned "ELOx:", where x is the 
// value of devNum. It is slightly faster to provide the name as an argument 
// rather than having it synthesized.

///////////////////////////////////////////////////////////////////////////////
namespace EloApi 
{

EloDevApi T_EloApi *newDevList( int *dimList = 0, int *cnt = 0 );

int T_EloApi forEach( bool (*callBack)( UCHAR devNum, TCHAR *name ) );
int T_EloApi rotateAll( USHORT rotateCmd );
int T_EloApi clearAllTouch( void );

// GenConEloDevId is a generic constructor for classes derived from EloDevId. 
// Wherever this macro is used, an inline constructor for the given class is 
// defined (not just declared-- see {} at end). This means that that the class 
// can have no additional initialization beyond the basic devNum and name, which
// are passed through to the EloDevId base class constructor. The class T 
// constructor exists only to provide a means to invoke the EloDevId 
// constructor.
#define GenConEloDevId(T) \
    T( UCHAR devNum = 0, TCHAR *name = 0 ) : EloDevId( devNum, name ) {};

// GenSetGet generates a standard setGet prototype and inline set and get 
// function definitions for the command types that fit this exact pattern.
#define GenSetGet \
    bool setGet( void ); \
    bool set( void ) { mIoc.mOp = IocSet; return setGet(); } \
    bool get( void ) { mIoc.mOp = IocGet; return setGet(); } 

//-------------------- SIMPLE COMMAND --------------------------------------
// Any command that can be passed to the driver as an ioctl code plus a single 
// ULONG argument, which the driver will also use to return a value (if any) 
// can be effected by an instance of this class without having to define a 
// unique argument transfer class. 
//
// Inline Command class functions afford a little simplification. It is also 
// possible to call the send function directly from a Command object, passing 
// arbitrary values. 
// Whether an inline function or direct send is invoked, the communication 
// always begins by creating a Command object:
// EloApi::Command cmd( devNum, devName ); 
// Direct and inline function calls through this object are equivalent. e.g.
// cmd.send( IOCTL_ELO_WRITECAL, WRITECAL_REG + WRITECAL_CONTROLLER ); = 
// cmd.saveCal( WRITECAL_REG + WRITECAL_CONTROLLER ); 
// 
// Simple commands: 
// IOCTL_ELO_ON_HOLD
// IOCTL_ELO_SOUND
// IOCTL_ELO_ON_TOUCH
// IOCTL_ELO_TOUCH
// IOCTL_ELO_ROTATE
// IOCTL_ELO_GETLINKTYPE
// IOCTL_ELO_APIDEVID
// IOCTL_ELO_SAVECAL
// IOCTL_ELO_DEBUGVAL 
// .........................................................................
class T_EloApi Command : public EloDevId
{
public:
    ULONG send( DWORD ioctl, ULONG arg );
    GenConEloDevId( Command ); // Constructor 

    ULONG onHold( ULONG onHold ) // OnHold = ON_HOLD_EVENT...ON_HOLD_NO_AUTOUNTOUCH        
    {	return send( IOCTL_ELO_ON_HOLD, onHold ); } 

    ULONG sound( ULONG touchSound ) // TouchSound = TOUCH_SOUND_OFF...IS_TOUCH_SOUND_ON
    {	return send( IOCTL_ELO_SOUND, touchSound ); } 

    USHORT setOnTouch( USHORT onTouch ) // OnTouch = ON_TOUCH_NOTHING...ON_TOUCH_UP		
	{	return USHORT( send( IOCTL_ELO_ON_TOUCH, onTouch + ( ON_TOUCH_SET << 16 ))); }

    USHORT getOnTouch( void )   	
	{	return USHORT( send( IOCTL_ELO_ON_TOUCH, ON_TOUCH_GET << 16 )); }

    void clearTouch( void )
    {   send( IOCTL_ELO_TOUCH, 0 ); }

    ULONG rotate( ULONG rotateCmd ) // 0 = 0, 1 = 90, 2 = 180, 3 = 270
	{	return send( IOCTL_ELO_ROTATE, rotateCmd ); }

    ULONG getLinkType( void ) // Returns DevLink = LINK_UNKNOWN, LINK_SERIAL, or LINK_USB		
	{	return send( IOCTL_ELO_GETLINKTYPE, 0 ); }

// Set/Get device ID for posting shared global events. ID is devIdCmd LOWORD.
// HIWORD is the command: DEVIDCMD_WR, DEVIDCMD_RESET, or DEVIDCMD_RD. In
// all cases, the resulting driver ID is returned.
    USHORT apiDevId( ULONG devIdCmd ) // DevIdCmd = DEVIDCMD_WR...DEVIDCMD_RD
	{	return LOWORD( send( IOCTL_ELO_APIDEVID, devIdCmd )); }

    USHORT writeApiDevId( USHORT id )
	{	return LOWORD( send( IOCTL_ELO_APIDEVID, (DEVIDCMD_WR << 16) + id )); }

    USHORT readApiDevId( void )
	{	return LOWORD( send( IOCTL_ELO_APIDEVID, DEVIDCMD_RD << 16 )); }

    USHORT resetApiDevId( void )
	{	return LOWORD( send( IOCTL_ELO_APIDEVID, DEVIDCMD_RESET << 16 )); }
    
    ULONG saveCal( ULONG saveCalTo ) // SaveCalTo = SAVECAL_DEFAULT...SAVECAL_EDGES	
	{	return send( IOCTL_ELO_SAVECAL, saveCalTo ); }

    ULONG debugVal( ULONG toDvr )                 
    {	return send( IOCTL_ELO_DEBUGVAL, toDvr ); } 
};

// EloApi library function EloApi::sendCmd simplfies caller by instantiating 
// its own Command object and invoking the send method. For multiple commands 
// sent to one device or to multiple devices using EloDevId::setDevice to change
// the device selection, better performance results from the caller making its 
// own longer-lived Command object. 
// The name argument is not required but providing it improves performance.
// Note context difference e.g:
// EloApi::Command.send vs. EloApi::sendCmd
// EloApi::Command.getLinkType vs. EloApi::getLinkType.
ULONG T_EloApi sendCmd( DWORD ioctl, ULONG arg, UCHAR devNum, TCHAR *name = 0  ); 

// Inline EloApi:: shell functions over sendCmd. 
inline ULONG setOnHold( ULONG onHold, UCHAR devNum, TCHAR *name = 0 )
{ return sendCmd( IOCTL_ELO_ON_HOLD, onHold, devNum, name ); }

inline ULONG disableHold( UCHAR devNum, TCHAR *name = 0 )
{ return sendCmd( IOCTL_ELO_ON_HOLD, ON_HOLD_GLOBAL, devNum, name ); }

inline ULONG disableHoldTemp( UCHAR devNum, TCHAR *name = 0 )
{ return sendCmd( IOCTL_ELO_ON_HOLD, ON_HOLD_GLOBAL+ON_HOLD_TEMP, devNum, name ); }

inline ULONG allowHold( UCHAR devNum, TCHAR *name = 0 )
{ return sendCmd( IOCTL_ELO_ON_HOLD, ON_HOLD_GLOBAL+ON_HOLD_EVENT, devNum, name ); }

inline ULONG setSound( ULONG touchSound, UCHAR devNum, TCHAR *name = 0 )
{ return sendCmd( IOCTL_ELO_SOUND, touchSound, devNum, name ); }

inline USHORT setOnTouch( USHORT onTouch, UCHAR devNum, TCHAR *name = 0 )
{ return (USHORT)sendCmd( IOCTL_ELO_ON_TOUCH, onTouch + ( ON_TOUCH_SET << 16 ), 
                  devNum, name ); }

inline ULONG setCleaningMode( UCHAR devNum, TCHAR *name = 0 )
{ return setOnTouch( ON_TOUCH_SYSMOUSE | ON_TOUCH_CLEANING, devNum, name ); }

inline USHORT getOnTouch( UCHAR devNum, TCHAR *name = 0 )
{ return (USHORT)sendCmd( IOCTL_ELO_ON_TOUCH, ON_TOUCH_GET << 16, devNum, name ); }

inline void clearTouch( UCHAR devNum, TCHAR *name = 0 )
{ sendCmd( IOCTL_ELO_TOUCH, 0, devNum, name ); }

inline ULONG rotate( ULONG rotateCmd, UCHAR devNum, TCHAR *name = 0 ) 		
{ return sendCmd( IOCTL_ELO_ROTATE, rotateCmd, devNum, name ); }

inline USHORT getLinkType( UCHAR devNum, TCHAR *name = 0 ) 
{ return (USHORT)sendCmd( IOCTL_ELO_GETLINKTYPE, 0, devNum, name ); }

inline USHORT setApiDevId( ULONG devIdCmd, UCHAR devNum, TCHAR *name = 0 ) 
{ return LOWORD( sendCmd( IOCTL_ELO_APIDEVID, devIdCmd, devNum, name )); }

inline USHORT writeApiDevId( USHORT id, UCHAR devNum, TCHAR *name = 0 )
{ return LOWORD( sendCmd( IOCTL_ELO_APIDEVID, (DEVIDCMD_WR << 16) + id,	devNum, name )); }

inline USHORT readApiDevId( UCHAR devNum, TCHAR *name = 0 )
{ return LOWORD( sendCmd( IOCTL_ELO_APIDEVID, DEVIDCMD_RD << 16, devNum, name )); }

inline USHORT resetApiDevId( UCHAR devNum, TCHAR *name = 0 )
{ return LOWORD( sendCmd( IOCTL_ELO_APIDEVID, DEVIDCMD_RESET << 16, devNum, name )); }

inline ULONG saveCal( ULONG saveCalTo, UCHAR devNum, TCHAR *name = 0 ) 	
{ return sendCmd( IOCTL_ELO_SAVECAL, saveCalTo, devNum, name ); }

inline ULONG debugVal( ULONG toDvr, UCHAR devNum, TCHAR *name = 0 )                 
{ return sendCmd( IOCTL_ELO_DEBUGVAL, toDvr, devNum, name ); } 


// -------------------------- IOCTL_ELO_CFGITEM -----------------------------
// See eloIoctl.h: class IocCfgItem
class T_EloApi CmdCfgItem : public EloDevId
{
public:
    IocCfgItem mIoc;
    GenConEloDevId( CmdCfgItem );
    GenSetGet
};

// setCfgItem and getCfgItem create a local CmdCfgItem using devNum and name 
// and then use this to send IOCTL_ELO_CFGITEM command with item and dat 
// parameters to the driver. The application could do this itself.
// item is an EloRegItemId defined in eloRegNames.h.
void T_EloApi setCfgItem( int item, DWORD dat, UCHAR devNum, TCHAR *name = 0 );
DWORD T_EloApi getCfgItem( int item, UCHAR devNum, TCHAR *name = 0 );

//---------------------IOCTL_ELO_HOLD_PARMS---------------------------------------
// This is not IOCTL_ON_HOLD, which uses generic command
// See eloIoctl.h: class IocHoldParms
class T_EloApi HoldParms : public EloDevId
{
public:
    IocHoldParms mIoc; 
    GenConEloDevId( HoldParms );
    bool send( void );
};

// IOCTL_ELO_ON_HOLD is Simple Command
// IOCTL_ELO_SOUND is Simple Command
// IOCTL_ELO_ON_TOUCH is Simple Command

//----------------------- IOCTL_ELO_MOUSELIMITS -------------------------------
// See eloIoctl: class IocMouseLimits 
class T_EloApi CmdMouseLimits : public EloDevId
{
    bool setOrGet( void );
public:
    IocMouseLimits mIoc;
    GenConEloDevId( CmdMouseLimits );
    bool get( void )    { mIoc.mOp = EloLimitGet; return setOrGet(); }
    bool setRel( void ) { mIoc.mOp = EloLimitSetRel; return setOrGet(); }
    bool setAbs( void ) { mIoc.mOp = EloLimitSetAbs; return setOrGet(); }
};

// -------------------------- IOCTL_ELO_DEVCFG -----------------------------
// See eloIoctl: class IocDevParms
class T_EloApi CmdDevCfg : public EloDevId
{
public:
    IocDevCfg mIoc;
    GenConEloDevId( CmdDevCfg );
    GenSetGet
};

//--------------------- IOCTL_ELO_REGKEYNAME ----------------------------------
// Get or set this device's registry key name. Setting is only for testing. Get
// provides the name for reading and writing the registry settings. The key is
// always under HKLM. Typically, applications do not use CmdRegKeyName directly
// but through getDevRegKeyName, which is easier to use and less likely to 
// change in the future.
class T_EloApi CmdRegKeyName : public EloDevId 
{
public:
    IocRegKeyName mIoc;
    GenConEloDevId( CmdRegKeyName );
	GenSetGet
};
//-----------------------------------------------------------------------------
// Function:	getDevRegKeyName   
// Purpose:		Writes the full name of the registry key (under HKLM) for the 
// given device to the given destination. This queries the driver to determine 
// the key currently in use and it may also manipulate the return value to 
// produce the full key name.
// Returns:		true if able to produce the key name.
// Arguments:	
//- TCHAR *outName is the destination supplied by the caller. This must contain
// MAX_ELOREGKEYNAME + 1 elements.
//- UCHAR devNum is the device number. Any value > 0 is legal but may not 
// necessarily refer to an attached device.
//- TCHAR *devName is optional name, e.g. "ELO1:", which overrides devNum.
bool T_EloApi getDevRegKeyName( TCHAR *outName, UCHAR devNum, TCHAR *devName = 0 );

// -------------------IOCTL_ELO_SMARTSET---------------------------------------
// See eloIoctl: class IocSmartSet
class T_EloApi CmdSmartSet : public EloDevId
{
public:
    IocSmartSet mIoc;
    GenConEloDevId( CmdSmartSet );
    bool mDo( void );
};
bool T_EloApi canDevTalk(  UCHAR devNum, TCHAR *name = 0 ); // SmartSet Ack query. 

//-------------------------- IOCTL_ELO_GETPOINT -------------------------------
// See eloIoctl: class IocGetPt 
class T_EloApi CmdGetPt : public EloDevId
{
public:
    IocGetPt mIoc;       
    GenConEloDevId( CmdGetPt );
    bool get( void );
};

// ------------------------ IOCTL_ELO_CAL3DAT ---------------------------------
// See eloIoctl: class IocCal3Dat
class T_EloApi CmdCal3Dat : public EloDevId
{   
public:
    IocCal3Dat mIoc;             
    GenConEloDevId( CmdCal3Dat );
    GenSetGet
};

//----------------------- IOCTL_ELO_EDGECAL ---------------------------------
// See eloIoctl: class IocEdgeCal 
class T_EloApi CmdEdgeCal : public EloDevId
{
public:
    IocEdgeCal *mIoc; // Pointer because IocEdgeCal is extensible.
    GenConEloDevId( CmdEdgeCal );
    bool set( void );
    bool get( void );
    bool getNoPts( IocEdgeCal *ec ) { 
        mIoc = ec; mIoc->mSize = sizeof( CmdEdgeCal ); return get(); }
};

// ------------------------- IOCTL_ELO_LINCOR ----------------------------------
// See eloIoctl.h: class IocLinCor

class T_EloApi CmdLinCor : public EloDevId
{   
    bool setOrGet( void );
public:
    IocLinCor mIoc;          
    GenConEloDevId( CmdLinCor );
    bool get( void )		{ mIoc.mOp = LINCOR_GET; return setOrGet(); }
    bool setOn( void )		{ mIoc.mOp = LINCOR_SET_ON; return setOrGet(); }
    bool setAdj( void )		{ mIoc.mOp = LINCOR_SET_ADJ; return setOrGet(); }
    bool setOnAdj( void )	{ mIoc.mOp = LINCOR_SET_ON+LINCOR_SET_ADJ; 
								return setOrGet(); }
};

// ------------------------- IOCTL_ELO_ACCEL ----------------------------------
// See eloIoctl.h: class IocAccel
class T_EloApi CmdAccel : public EloDevId
{   
public:
    IocAcc mIoc;           
    GenConEloDevId( CmdAccel );
    bool setOrGet( void );
    bool get( void )		{ mIoc.mOp = GET_ACCEL; return setOrGet(); }
    bool set( void )		{ mIoc.mOp = 
		SET_ACCEL_METHOD + SET_ACCEL_RATE + SET_ACCEL_POS; return setOrGet(); }
    bool setMethod( void )	{ mIoc.mOp = SET_ACCEL_METHOD; return setOrGet(); }
    bool setRate( void )	{ mIoc.mOp = SET_ACCEL_RATE; return setOrGet(); }
    bool setPos( void )		{ mIoc.mOp = SET_ACCEL_POS; return setOrGet(); }
};

// ------------------------- IOCTL_ELO_CORNERACC ------------------------------
// See eloIoctl.h: class IocCornerAcc
class T_EloApi CmdCornerAcc : public EloDevId
{
public:
    bool setOrGet( void );
    IocCornerAcc mIoc;           
    GenConEloDevId( CmdCornerAcc );
    bool get( void )		{ mIoc.mOp = CACC_GET; return setOrGet(); }
    bool setOff( void )		{ mIoc.mOp = CACC_SET_ENABLE; return setOrGet(); }
    bool setAcc( void )		{ mIoc.mOp = CACC_SET_ACC; return setOrGet(); }
    bool setOn( void )		{ mIoc.mOp = CACC_SET_ENABLE+CACC_ON; 
								return setOrGet(); }
    bool setOnAcc( void )	{ mIoc.mOp = CACC_SET_ACC+CACC_SET_ENABLE+CACC_ON; 
		                        return setOrGet(); }
};

}   // namespace EloApi 


#endif // #ifndef ELOAPI_H
